function generateCA(commonName) {
    const CA = {};

    // Create X.509 certificate object
    const cert = forge.pki.createCertificate();

    // Generate 4096-bit RSA keypair
    const keys = forge.pki.rsa.generateKeyPair(2048);
    CA.PublicKey = keys.PublicKey;
    CA.PrivateKey = keys.PrivateKey;

    // Add public key to certificate
    cert.publicKey = keys.publicKey;

    // Set serial number
    cert.serialNumber = "0";

    // Set certificate start date to NOW
    cert.validity.notBefore = new Date();

    // Set certificate expiration date to NOW + ttl seconds
    cert.validity.notAfter = new Date();
    cert.validity.notAfter.setSeconds(cert.validity.notAfter.getSeconds() + 60);

    // Set issuer and subject attributes
    const attrs = [
        {
            name: "organizationName",
            value: "Wizardry and Steamworks"
        }, {
            name: "organizationalUnitName",
            value: "Corrade"
        }, {
            name: "emailAddress",
            value: "office@grimore.org"
        }, {
            name: "commonName",
            value: commonName
        }
    ];

    cert.setSubject(attrs);
    cert.setIssuer(attrs);

    cert.setExtensions([
        {
            name: "basicConstraints",
            cA: true
        }, {
            name: "keyUsage",
            keyCertSign: true,
            digitalSignature: true,
            nonRepudiation: true,
            keyEncipherment: true,
            dataEncipherment: true
        }, {
            name: "extKeyUsage",
            serverAuth: true,
            clientAuth: true,
            codeSigning: true,
            emailProtection: true,
            timeStamping: true
        }, {
            name: "nsCertType",
            client: true,
            server: true,
            email: true,
            objsign: true,
            sslCA: true,
            emailCA: true,
            objCA: true
        }, {
            name: "subjectKeyIdentifier"
        }
    ]);

    // Create SHA1 hash object (SHA256 not supported by Firefox)
    const hash = forge.md.sha1.create();

    // Self-sign certificate
    cert.sign(keys.privateKey, hash);
    CA.Certificate = cert;

    // convert a Forge private key to an ASN.1 RSAPrivateKey
    const rsaPrivateKey = forge.pki.privateKeyToAsn1(keys.privateKey);

    // wrap an RSAPrivateKey ASN.1 object in a PKCS#8 ASN.1 PrivateKeyInfo
    const privateKeyInfo = forge.pki.wrapRsaPrivateKey(rsaPrivateKey);
    CA.PrivateKeyInfo = privateKeyInfo;

    return CA;
}

function generateServerCertificate(CACertificate, CAPrivateKey) {
    // Wrap ASN.1 with private key into empty password PKCS12-container.
    // (Chrome/Firefox can only handle 3DES encryption)
    const pkcs12 = forge.pkcs12.toPkcs12Asn1(CAPrivateKey,
        [CACertificate],
        "",
        {
            algorithm: "3des"
        });

    // Convert PKCS12 to DER format
    const pkcs12Der = forge.asn1.toDer(pkcs12).getBytes();

    const asn1 = forge.asn1;
    console.log(asn1.prettyPrint(pkcs12));

    // Return Base64-encoded PKCS12 certificate.
    return forge.util.encode64(pkcs12Der);
}